document.addEventListener("DOMContentLoaded", () => {
  loadWatchlistItems()
})

function loadWatchlistItems() {
  const container = document.getElementById("watchlistCardsContainer")
  if (!container) {
    console.error("Watchlist container not found")
    return
  }
  container.innerHTML = '<p class="loading-message">Chargement de votre liste "À venir"...</p>'

  fetch("api/watchlist.php")
    .then((response) => {
      if (!response.ok) {
        throw new Error(`HTTP error! status: ${response.status}`)
      }
      return response.json()
    })
    .then((result) => {
      if (result.success && result.data) {
        renderWatchlistCards(result.data)
      } else if (result.success && result.data.length === 0) {
        container.innerHTML = '<p class="empty-message">Votre liste "À venir" est vide.</p>'
      } else {
        container.innerHTML = `<p class="error-message">Erreur: ${result.error || "Impossible de charger la liste."}</p>`
        console.error("Failed to load watchlist items:", result.error)
      }
    })
    .catch((error) => {
      container.innerHTML = `<p class="error-message">Erreur de connexion: ${error.message}</p>`
      console.error("Error fetching watchlist items:", error)
    })
}

function renderWatchlistCards(items) {
  const container = document.getElementById("watchlistCardsContainer")
  container.innerHTML = "" // Clear loading message or previous content

  if (items.length === 0) {
    container.innerHTML = '<p class="empty-message">Votre liste "À venir" est vide.</p>'
    return
  }

  items.forEach((item) => {
    const cardItem = document.createElement("div")
    cardItem.className = "card-item"
    cardItem.onclick = () => viewPerformance(item.performance_id)

    const posterUrl = item.poster_url || "assets/images/placeholder.jpg"
    const title = item.performance_title || "Titre inconnu"
    const avgRating = Number.parseFloat(item.avg_rating || 0).toFixed(1)
    const reviewCount = item.review_count || 0

    let eventDateFormatted = "Date non spécifiée"
    if (item.event_date) {
      try {
        const dateObj = new Date(item.event_date)
        eventDateFormatted = `Le ${dateObj.toLocaleDateString("fr-FR", { day: "2-digit", month: "2-digit", year: "numeric" })} à ${dateObj.toLocaleTimeString("fr-FR", { hour: "2-digit", minute: "2-digit" })}`
      } catch (e) {
        console.warn("Invalid date format for item:", item, e)
        eventDateFormatted = item.event_date // fallback to raw string
      }
    }

    cardItem.innerHTML = `
      <div class="card" style="background-image: url('${posterUrl}')"></div>
      <div class="card-info">
        <div class="title">${escapeHTML(title)}</div>
        <div class="watchlist-event-date">${eventDateFormatted}</div>
        <div class="rating">
          ★ ${avgRating} (${reviewCount})
        </div>
        <button class="btn-remove-watchlist" data-performance-id="${item.performance_id}" data-event-date="${item.event_date}" title="Retirer de la liste">
          🗑️ Retirer
        </button>
      </div>
    `
    container.appendChild(cardItem)
  })

  // Add event listeners for remove buttons
  document.querySelectorAll(".btn-remove-watchlist").forEach((button) => {
    button.addEventListener("click", (e) => {
      e.stopPropagation() // Prevent card click
      const performanceId = button.dataset.performanceId
      const eventDate = button.dataset.eventDate
      removeFromWatchlist(performanceId, eventDate, button.closest(".card-item"))
    })
  })
}

function removeFromWatchlist(performanceId, eventDate, cardElement) {
  if (!confirm('Êtes-vous sûr de vouloir retirer cet élément de votre liste "À venir" ?')) {
    return
  }

  fetch("api/watchlist.php", {
    method: "POST",
    headers: {
      "Content-Type": "application/json",
    },
    body: JSON.stringify({
      performance_id: Number.parseInt(performanceId),
      event_date: eventDate, // API expects YYYY-MM-DD HH:MM:SS
    }),
  })
    .then((response) => response.json())
    .then((result) => {
      if (result.success && result.action === "removed") {
        if (typeof showNotification === "function") {
          showNotification(result.message || "Retiré de la liste avec succès.", "success")
        } else {
          alert(result.message || "Retiré de la liste avec succès.")
        }
        if (cardElement) {
          cardElement.remove() // Remove the card from the DOM
          // Check if list is now empty
          const container = document.getElementById("watchlistCardsContainer")
          if (container && container.children.length === 0) {
            container.innerHTML = '<p class="empty-message">Votre liste "À venir" est vide.</p>'
          }
        } else {
          loadWatchlistItems() // Fallback to reload all items
        }
      } else {
        if (typeof showNotification === "function") {
          showNotification(result.error || "Erreur lors de la suppression.", "error")
        } else {
          alert(result.error || "Erreur lors de la suppression.")
        }
      }
    })
    .catch((error) => {
      console.error("Error removing from watchlist:", error)
      if (typeof showNotification === "function") {
        showNotification("Erreur de connexion.", "error")
      } else {
        alert("Erreur de connexion.")
      }
    })
}

function viewPerformance(id) {
  window.location.href = `performance.php?id=${id}`
}

function escapeHTML(str) {
  const div = document.createElement("div")
  div.appendChild(document.createTextNode(str))
  return div.innerHTML
}
